<?php
/**
 * Plugin Name: Netstorm Extra
 * Description: Custom Elementor extension.
 * Plugin URI:  https://theme-land.com
 * Version:     1.0.0
 * Author:      Themeland
 * Author URI:  https://theme-land.com
 * Text Domain: netstorm
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Main Netstorm Extra Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */
final class Netstorm_Elements {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string The plugin version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */
	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.0';

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Netstorm_Elements The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Netstorm_Elements An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {

		add_action( 'plugins_loaded', [ $this, 'on_plugins_loaded' ] );

	}

	/**
	 * Load Textdomain
	 *
	 * Load plugin localization files.
	 *
	 * Fired by `init` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function i18n() {

		load_plugin_textdomain( 'netstorm' );

	}

	/**
	 * On Plugins Loaded
	 *
	 * Checks if Elementor has loaded, and performs some compatibility checks.
	 * If All checks pass, inits the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function on_plugins_loaded() {

		if ( $this->is_compatible() ) {
			add_action( 'elementor/init', [ $this, 'init' ] );
		}

	}

	/**
	 * Compatibility Checks
	 *
	 * Checks if the installed version of Elementor meets the plugin's minimum requirement.
	 * Checks if the installed PHP version meets the plugin's minimum requirement.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function is_compatible() {

		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return false;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return false;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return false;
		}

		return true;

	}

	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {
	
		$this->i18n();

		// Add Plugin actions
		add_action( 'elementor/widgets/widgets_registered', [ $this, 'init_widgets' ] );

        // Register Custom Category
        add_action( 'elementor/elements/categories_registered', [ $this, 'add_elementor_category' ] );

		// Register Widget Styles
		add_action( 'elementor/frontend/after_enqueue_styles', [ $this, 'widget_styles' ] );

		// Register Widget Scripts
		add_action( 'elementor/frontend/after_register_scripts', [ $this, 'widget_scripts' ] );

		// Included Helper file
		require_once( __DIR__ . '/includes/helper-function.php' );

		// Included Framework file
		require_once( __DIR__ . '/includes/framework/redux-core/framework.php' );
		require_once( __DIR__ . '/includes/framework/sample/config.php' );

		// Register Custom Posts
		require_once( __DIR__ . '/includes/custom-posts/portfolio.php' );
		require_once( __DIR__ . '/includes/custom-posts/auction.php' );
		require_once( __DIR__ . '/includes/custom-posts/collection.php' );
		require_once( __DIR__ . '/includes/custom-posts/seller.php' );
		require_once( __DIR__ . '/includes/custom-posts/vendor.php' );

	}

	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init_widgets() {

		// Include Widget files
		require_once( __DIR__ . '/widgets/accordion.php' );
		require_once( __DIR__ . '/widgets/breadcrumb.php' );
		require_once( __DIR__ . '/widgets/section-heading.php' );
		require_once( __DIR__ . '/widgets/hero.php' );
		require_once( __DIR__ . '/widgets/top-seller.php' );
		require_once( __DIR__ . '/widgets/live-auctions.php' );
		require_once( __DIR__ . '/widgets/explore.php' );
		require_once( __DIR__ . '/widgets/collections.php' );
		require_once( __DIR__ . '/widgets/authors.php' );
		require_once( __DIR__ . '/widgets/author-profile.php' );
		require_once( __DIR__ . '/widgets/item-details.php' );
		require_once( __DIR__ . '/widgets/activity.php' );
		require_once( __DIR__ . '/widgets/wallet-connect.php' );
		require_once( __DIR__ . '/widgets/create.php' );
		require_once( __DIR__ . '/widgets/login.php' );
		require_once( __DIR__ . '/widgets/signup.php' );
		require_once( __DIR__ . '/widgets/icon-box.php' );
		require_once( __DIR__ . '/widgets/work.php' );
		require_once( __DIR__ . '/widgets/help-center.php' );
		require_once( __DIR__ . '/widgets/posts.php' );
		require_once( __DIR__ . '/widgets/post-details.php' );
		require_once( __DIR__ . '/widgets/form.php' );

		//tony edit
		require_once( __DIR__ . '/widgets/explore-products.php' );
		require_once( __DIR__ . '/widgets/my-wallet.php' );


		// Register widget
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Accordion() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Breadcrumb() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Section_Heading() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Hero() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Top_Seller() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Live_Auctions() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Explore() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Collections() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Authors() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Author_Profile() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Item_Details() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Activity() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Wallet_Connect() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Create() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Login() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Signup() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Icon_Box() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Work() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Help_Center() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Posts() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Post_Details() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Form() );


		//tony edit
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \Explore_Products() );
		\Elementor\Plugin::instance()->widgets_manager->register_widget_type( new \My_Wallet() );

		
	}

    /**
	 * Init Category
	 */
    public function add_elementor_category( $elements_manager ) {

        $elements_manager->add_category(
            'netstorm-elements',
            [
                'title' => __( 'Netstorm Elements', 'netstorm' ),
                'icon' => 'fa fa-plug',
            ]
        );

    }

	/**
	 * Register Widget Styles
	 */
	public function widget_styles() {

		wp_register_style( 'netstorm-element', plugins_url( '/assets/css/netstorm-element.css', __FILE__ ) );

	}

	/**
	 * Register Widget Scripts
	 */
	public function widget_scripts() {

		wp_register_script( 'netstorm-element', plugins_url( '/assets/js/netstorm-element.js', __FILE__ ), [ 'jquery' ] );

	}
	
	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'netstorm' ),
			'<strong>' . esc_html__( 'Netstorm Extra', 'netstorm' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'netstorm' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'netstorm' ),
			'<strong>' . esc_html__( 'Netstorm Extra', 'netstorm' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'netstorm' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'netstorm' ),
			'<strong>' . esc_html__( 'Netstorm Extra', 'netstorm' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'netstorm' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

}

Netstorm_Elements::instance();